#ifndef LLF_RND_H
#define LLF_RND_H

/*
* All the includes that are needed for code using this module to
* compile correctly should be #included here.
*/

#include "CE2_public.h"

#ifdef __cplusplus
extern "C"
{
#endif

/*
*  Object name     :  LLF_RND.h
*  State           :  %state%
*  Creation date   :  29.05.2007
*  Last modified   :  %modify_time%
*/
/** @file
* \brief contains declarations of low level functions for RND.
*
*  \version LLF_RND.h#1:cinc:1
* \author Yermalayeu Ihar
* \remarks Copyright (C) 2007 by Discretix Technologies Ltd. All rights reserved.
*/

/************************ Defines *****************************/
/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Structs *****************************/
/************************ Public Variables ********************/
/************************ Public Functions ********************/

#ifdef RND_FIPS_PUB_186_2

/**
****************************************************************
* Function Name: 
*  LLF_RND_GenerateVector
*
*  @param[in] RndSize - The size of random vector that is required.
*  @param[in/out] Output_ptr - The output vector.
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - on failure a value from CE2_error.h:
*
* \brief \b 
* Description:
*  The LLF_RND_GenerateVector function generates a random vector using
*  the FIPS-PUB 186-2 standard, Appendix 3.
*  The random function is based on the HASH SHA1 and works as follows:
*  1. Loads the secret key stored internally into a global variable.
*  2. Executes the following steps according to the number of hash result 
*     blocks required (RndSize/20).
*    a. Executes a SHA1 hash on the current key
*       (if the key is less than 512 bytes, adds zeros to it).
*    b. Stores the result to the data 
*    c. The next key is 1 ^ key ^ result. This is the new key.
*
*  \b 
* Algorithm:
*  -# 
***************************************************************/
CE2Error_t LLF_RND_GenerateVector(DxUint16_t  RndSize,
  DxUint8_t  *Output_ptr);

/**
****************************************************************
* Function Name: 
*  LLF_RND_ResetSeed
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - on failure a value from CE2_error.h:
*
* \brief \b 
* Description:
*  The LLF_RND_ResetSeed resets the SEED generated by the low level
*  (the hardware or operating system service on software).
*
*  \b 
* Algorithm:
*  -#  
***************************************************************/
CE2Error_t LLF_RND_ResetSeed(void);

/**
****************************************************************
* Function Name: 
*  LLF_RND_GenerateVectorInRange
*
*  @RndSize[in] - The size of random vectore that is required.
*  @MaxVect_ptr[in]  - The pointer to vector defines high limit of random vector.
*  @RndVect_ptr[in/out] The output vector.
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - on failure a value from CE2_error.h:
*
* \brief \b 
* Description:
*  This function generates a random vector RndVect in range:
*  1 < RndVect < MaxVect,   using the FIPS-PUB 186-2 standard appendex 3 :
*  The function performs the following:
*    1. Calls the CE2_RND_GenerateVector() function for generating random vector 
*       RndVect of size in bytes.
*    2. Zeroes extra bits of RndVect.
*    3. Compares RndVect with 1 and MaxVect. If condition  1 < RandVect < MaxVect
*       is not satisfied goues to step 1.
*
*  \b 
* Algorithm:
*  -#  
***************************************************************/
CE2Error_t LLF_RND_GenerateVectorInRange(DxUint32_t RndSizeInBits, 
  DxUint8_t *MaxVect_ptr, 
  DxUint8_t *RndVect_ptr);


#elif RND_FIPS_SP_800_90

/************************ Defines *****************************/

#define LLF_RND_DEFAULT_DATA_SIZE_BITS		128
#define LLF_RND_ADDITIONAL_DATA_SIZE_BITS	256

#define LLF_RND_DEFAULT_DATA_SIZE			(LLF_RND_DEFAULT_DATA_SIZE_BITS / 8)
#define LLF_RND_ADDITIONAL_DATA_SIZE		(LLF_RND_ADDITIONAL_DATA_SIZE_BITS / 8)

/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Structs *****************************/
/************************ Public Variables ********************/
/************************ Public Functions ********************/

/**
****************************************************************
* Function Name: 
*  LLF_RND_Instantiation
*
* Input:
*  None
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - on failure a value from CE2_error.h:
*
* \brief \b 
* Description:
*  Initializes RND engine according to FIPS-SP 800-90:
*  - CCSystemSpec_RNG v1.2 [2.3.2.2 Instantiation];
*  - RNG_SW_SDDv0.3.
*
*  \b 
* Algorithm:
*  - 1. Collect 128 bits of entropy (from entropy_input);
*  - 2. Computing the seed_material value;
*  - 3. CTR_DRBG_Update;
*  - 4. Increment the V variable, using full 128-bit arithmetic;
*  - 5. Reset the additional input flag.
***************************************************************/
CE2Error_t LLF_RND_Instantiation(void);

/**
****************************************************************
* Function Name: 
*  LLF_RND_GenerateVector
*
* Input:
*  @param[in]     RndSize    - The size of random vector that is required;
*  @param[in/out] Output_ptr - The output vector.
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - on failure a value from CE2_error.h:
*
* \brief \b 
* Description:
*  Generates pseudo random data vector according to FIPS-SP 800-90:
*  - CCSystemSpec_RNG v1.2 [2.3.2.4 Random Number Generation];
*  - RNG_SW_SDDv0.3.
*
*  \b 
* Algorithm:
*  -# Verify Initial update of working state with additional input;
*  -# Generation of output bits;
*  -# Final update of the working state.
***************************************************************/
CE2Error_t LLF_RND_GenerateVector(DxUint16_t RndSize,		/* in */
								  DxUint8_t  *Output_ptr);	/* in/out */

/**
****************************************************************
* Function Name: 
*  LLF_RND_Reseeding
*
* Input:
*  None
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - on failure a value from CE2_error.h:
*
* \brief \b 
* Description:
*  Preforms RND engine reseeding according to FIPS-SP 800-90:
*  - CCSystemSpec_RNG v1.2 [2.3.2.3 Reseeding];
*  - RNG_SW_SDDv0.3.
*
*  \b 
* Algorithm:
*  - 1. Collect 128 bits of entropy;
*  - 2. Computation of the new 256-bit seed_material;
*  - 3. Encrypt the V register using the K register as key.
*       Increment V (using full 128-bit arithmetic);
*  - 4. XOR the result of step 3 with the result of step 2j,
*       and keep for future use as the next value for the K register;
*  - 5. Encrypt the V register (after it was incremented in step 3)
*       using the K register as key;
*  - 6. XOR the result of step 5 with the result of step 2k,
*       and store the result in the V register;
*  - 7. Copy the result from step 4 to the K register;
*  - 8. Increment the V register, using full 128-bit arithmetic;
*  - 9. Reset the additional input flag.

***************************************************************/
CE2Error_t LLF_RND_Reseeding(void);

/**
****************************************************************
* Function Name: 
*  LLF_RND_AddAdditionalInput
*
* Input:
*  @param[in] AdditionalInput     - Additional input data;
*  @param[in] AdditionalInputSize - Additional input data size.
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - on failure a value from CE2_error.h:
*
* \brief \b 
* Description:
*  Adds additional info vector to RND engine according to:
*  - CCSystemSpec_RNG v1.2;
*  - RNG_SW_SDDv0.3.
*
*  \b 
* Algorithm:
*  -# Set additional info data.
***************************************************************/
CE2Error_t LLF_RND_AddAdditionalInput(DxUint8_t  *AdditionalInput,		/* in */
									  DxUint16_t AdditionalInputSize);	/* in */

#endif

#ifdef __cplusplus
}
#endif

#endif /* LLF_RND_H */

